; *********************************************************************************************
; *********************************************************************************************
;
;	Einfaches Modell eines Datenloggers mit dem ATMega 8
; 	Am ADC0-Eingang liegt das Signal des Temperatursensors, der bereits mit dem Analogcomparator
; 	eingesetzt wurde.
;	Es knnen bis zu 240 Messungen mit einer Zeitbasis von 1 Sekunde bis zu 99 Min und 99 Sekunden
;	eingestellt werden (=6039 Sek. entspr. 14,3 Messungen pro Tag = 16,77 Tage Messdauer)
; 	
;	Autor: Jrgen Grzesina
;	Dateiname: adc.asm
; 	Stand: 25.02.2009
;
; *********************************************************************************************
; *********************************************************************************************
;
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; Deklarationen
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

 .equ mega8 = 1						; wir verwenden einen mega8 (wichtig fr die RS232-Routinen)
 .equ debuglevel = 0				; Einstellung verschiedener Testprogramme
	
; Wir verfolgen in diesem Projekt streng das Konzept von Includedateien
; Bei sehr umfangreichen Bibliotheksmodulen gibt es eine Headerdatei, die
; die Declarationen enthlt. In der Regel gengt es, Anpassungen dort vorzunehmen.
; Die eigentlichen Routinen werden am Schluss der Hauptdatei ADC.asm vor den Jobproceduren 
; eingebunden. Die Verfgbarkeit mancher Bibliotheksroutinen ist von der Verfgbarkeit  
; anderer abhngig. Zur Vermeidung von Fehlermeldungen erfolgt die Absicherung ber .if-Konstrukte
;
; Fr die Includes wird kein Listing erstellt
.nolist
.include "m8def.inc"
.include "arithmetik_header.asm"
.include "lcd-routines-bib_header.asm"
.include "rs232inc_header.asm"
; die Datei eeprom_512_inc.asm erfordert keinen Header
; die Datei adc_m8_inc.asm erfordert keinen Header
.list

	.equ led_port = portd
	.equ led_ddr = ddrd
	.equ led1 = 5					; Test LED an PORTD.5

	.def temp1 = r16
	.def temp2 = r17
	.def temp3 = r18
	.def flag = r19
		.equ fblinken = 0
		.equ fgetvalue = 1
	.def wahl = r20					; Ausgesuchter Menpunkt
		.equ mwelcome = 0 			; Die folgenden Werte sind keine Bits, sondern Zustnde
		.equ mSek = 1				; Sekunden Einstellung
		.equ mMin = 2				; Minuten  Einstellung
		.equ mInt = 3				; Intervall Berechnen/anzeigen
		.equ mMaxSamp = 4			; Obergrenze Werte Einstellung
		.equ mLoeschen = 5			; Werteliste lschen
		.equ mStopnGo = 6			; Reihe starten und stoppen
		.equ mSend = 7				; Werte senden
		.equ mLCDClear = 8			; LCD lschen
		.equ mKanal = 9				; Anzahl Messeingnge
		.equ mWerte = 10			; Werte anzeigen
		.equ mlast	= 11			; wenn "wahl" = mlast, wird auf 1 zurckgesetzt

	.def anzahlwerte = r21

	.def t1 = R22					; Tastenzhler
	.def t2 = R23					; fr Tastenentprellung 
	.def t3 = R24					; in der ISR von Timer 1

	.equ tastenport = portd			; Port, an den die Tasten angeschlossen sind
	.equ tastenddr	= ddrd			; dto DDR
	.equ tastenpin	= pind			; dto Eingabeport
		.equ taste1		= 4			; Portleitung fr Taste 1	-> schaltet Menu weiter
		.equ taste2		= 3			; Portleitung fr Taste 2	-> Go, Count up
		.equ taste3		= 2			; Portleitung fr Taste 3	-> Stop, count down
		.equ mns = 4				; Men noch nicht gestartet, Start mit Taste 1
		.equ keypressed = 7			; eine Taste wurde gedrckt, wird erst gesetzt, wenn mns=0
	.equ longtime   = 4				; Ab diesem Wert wird ein Langzeittastendruck registriert

	.def rKanaele = R25				; Anzahl von Kanaelen

	.def count_low = R26
	.def count_high = R27			; Zeit-Countdown-Zhler fr Intervalle
;
	.equ takt = 8000000
	.equ vwert = 800-1				; Wert fr ocr1a-Vergleichsregister

	.equ sec_preset = 2
	.equ min_preset = 0				; Vorbelegungen fr Sekunden und Minuten


.dseg
	grubs:							; Die kleinste Zeiteinheit	
	.byte 1							; ist ein 25 stel einer Hundertstelsekunde
	jiffies:						; chronischer Registermangel zwingt dazu,
	.byte 1							; manche Variablen in den SRAM-Speicher
	sekunden:						; auszusiedeln
	.byte 1							; Wir definieren hie Speicherstelle und Umfang
	minuten:
	.byte 1
	stunden:
	.byte 1

	zeit_low:
	.byte 1
	zeit_high:
	.byte 1

	maxzahl:						; Maximalzahl an Messwerten
	.byte 1

	dKanaele:						; Anzahl an verwendeten Messeingngen
	.byte 1

	dkanal:							; aktueller Kanal (bei Anzeige)
	.byte 1

	dawert:							; aktueller wertezeiger
	.byte 1
	dpointerl:						; 16-Bit-Zeiger auf Wertetabelle fr LCD-Ausgabe
	.byte 2
	.equ dpointerh = dpointerl  + 1
	dwerte:							; Anzahl gelesener Werte (bei Anzeige)
	.byte 1	

.eseg
	timebase_low:  .byte 1			; hnlich wie im SRAM werden Speicherstellen im
	timebase_high:  .byte 1			; EEPROM-Bereich reserviert
	number_of_values: .byte 2		; Anzahl gesampelter Werte
	.equ maximalwert = 240
	werteliste:
		w_low: .byte maximalwert
		w_high: .byte maximalwert

	kanaele: .byte 1				; Anzahl Messeingnge
	.equ maxkanal = 4				; grte Anzahl von simulatanen Messeingngen

.cseg
.org 0x0000
; Resetvektor
	rjmp start

.org 0x06; OC1Aaddr					; IRQ-Vektor fr Timer 1
	rjmp Timer1_oc1a				; Sprung zur Serviceroutine


.org 0x001A					; Reset-Einsprung
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; Vorbereitungen
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
start:
	ldi temp1, low(ramend)
	out spl,temp1			; stackpointer setzen
	ldi temp1, high(ramend)
	out sph,temp1			; stackpointer setzen

; rs232 initialisieren
	rcall serinit


; ADC initialisieren: Single Conversion, Vorteiler 64 -> 125 kHz Samplefrquenz bei 8  MHz
	rcall adc_init			; adc initialisieren
	ldi temp1, 1			; und mit einer ersten Messung beauftragen
	rcall adc_get			; Jetzt werden pro Wandlung nur noch 13 Takte gebraucht



; lcd initialisieren
	rcall lcd_init
	rcall lcd_clear
	rcall lcd_home
	ldi temp1, (1<<dispon) | (0<<curson) | (0<<cursbl)
	rcall lcd_cursor
	ldi wahl, mwelcome				; Men4 = Werte lschen - erstes dagestelltes Men wird 5
	ldi flag, 1<<mns				; lsche alle Merker, setzt "menu not started"
	rcall show_menu
	ldi wahl, mloeschen				; Auswahl des Mens 5 vorbereiten

; Timer1 fr System-IRQ einrichten
	; Zhler 1 fr die Erzeugung des Zweihundertstel-Sekundentakts vorbereiten
	ldi temp1, high(vwert)			; OCR1A setzen
	out ocr1ah, temp1				; wir lassen Timer1 bis 20000 zhlen
	ldi temp1, low(vwert)			; das schafft er 50 mal pro Sekunde bei 1 MHz Timertakt = Systemtakt/8
	out ocr1al, temp1				; ein Durchgang = 1 Jiffie = 1/50 s

	ldi temp1, (1<<wgm12) | (0<<cs12) | (1<<cs11) | (0<<cs10); CTC ein, Taktquelle = Systemtakt/8 ein
	out tccr1b, temp1
	ldi temp1, 0					; tccr1a alle Bits = 0
	out tccr1a, temp1

	ldi temp1, (1<<ocie1a)			; output compare 1A IRQ zulassen
	out timsk, temp1

	ldi temp1, 0
	sts grubs, temp1
	sts sekunden, temp1				; Zeitrechnung auf 0 stellen
	sts minuten, temp1
	sts stunden, temp1
	sts jiffies, temp1
	mov anzahlwerte,temp1			; Anzahl werte im Register auf Null
									; Werte im EEPROM bleiben unangetastet
	

	sei								; Interrupts zulassen
	
	sbi led_ddr, led1				; Leitung fr LED1 auf Ausgang
	cbi led_port, led1				; LED1 aus
	ldi yl, low(werteliste)
	ldi yh, high(werteliste)		; Adressenverwaltung fr EEPROM	
	ldi temp1, maximalwert			; Maximalanzahl Messwertpaare LOW/HIGH
	sts maxzahl, temp1				; voreinstellen

.if debuglevel == 1 
	; Zeit und Kanal Preset wird nur in der Testphase bentigt.
	rcall vorgabewerte
.endif

.if debuglevel == 2
	; EEPROM mit inhalt fllen
	ldi yl, low(werteliste)
	ldi yh, high(werteliste)		; Adressenverwaltung fr EEPROM	
	ldi zl, low (500)
	ldi zh, high(500)
nochmal:
	mov temp2, yl
	mov temp3, yh
	mov temp1, zl
	rcall eep_write
	adiw y, 1
	mov temp2, yl
	mov temp3, yh
	mov temp1,zh
	rcall eep_write
	adiw y,1
	adiw z,1
	inc anzahlwerte
	mov temp1, anzahlwerte
	ldi temp2, low(number_of_values)
	ldi temp3, high(number_of_values)
	rcall eep_write
	cpi anzahlwerte, 12
	brne nochmal
fertig_hier:
	ldi yl, low(werteliste)
	ldi yh, high(werteliste)		; Adressenverwaltung fr EEPROM	
	ldi temp1, 0
	mov anzahlwerte,temp1
.endif

; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; Hauptschleife
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
loop:
	sbrs flag, fgetvalue	; wurde ein neuer Messwert registriert?
	rjmp main				; nein, dann schauen wir mal nach Tastenbefehlen von der RS232
	rcall werte_sichern		; ja, neuer Wert, dann mssen wir erst mal den Wert sichern
Main:
	sbic ucsra, rxc			; wenn das RXC-Bit gesetzt ist, gibt es einen neuen Befehl von RS232
	rcall rs_befehl			

tastendruck:				; haben wir einen Tastendruck?
	sbrc flag, keypressed	; Bit kexpressed in Flag gesetzt? rcall berpringen
	rcall menu				; Taste decodieren
	in temp1, mcucr
	andi temp1, 0b00001111	; Highnibble lschen
	ori temp1, ((1<<SE) | (0<<sm2) | (0<<sm1) | (0<<sm0))
	out mcucr, temp1		; Sleepmode setzen
	sleep					; schlafen schicken
	rjmp loop				; Ist nicht der Fall, neuer Schleifendurchlauf

menu:
	push zl
	push zh

menu0:
	sbrs flag, taste1		; Taste 1  gedrckt, also neue Menanzeige
	rjmp menu1
	rcall show_menu			; Men mit der Nummer in "wahl" anzeigen
	cpi wahl, 1
	brne menu0_a
	; Sekundenvorgabe
	ldi temp1, low(timebase_low)
	ldi temp2, high(timebase_low)
	rcall eep_read			; Wert ist jetzt in temp1
	push temp1
	ldi temp1, 6
	ldi temp2, 1
	rcall lcd_position
	pop temp1
	rcall lcd_number
	rcall lcd_blank
	rjmp menu0_done
menu0_a:
	cpi wahl, 2
	brne menu0_b
	; Minutenvorgabe
	ldi temp1, low(timebase_high)
	ldi temp2, high(timebase_high)
	rcall eep_read			; Wert ist jetzt in temp1
	push temp1
	ldi temp1, 6
	ldi temp2, 1
	rcall lcd_position
	pop temp1
	rcall lcd_number
	rcall lcd_blank
	rjmp menu0_done
menu0_b:
	cpi wahl, 4
	brne menu0_c
	; Maximalwertvorgabe
	rcall show_samples
	rjmp menu0_done
menu0_c:
	cpi wahl, 9
	brne menu0_d
	; Kanalvorgabe
	ldi temp1, 6
	ldi temp2, 1
	rcall lcd_position
	rcall get_kanaele
	rcall lcd_number
	rcall lcd_blank
;	rjmp menu0_done

menu0_d:
	cpi wahl, 10
	brne menu0_e
	ldi temp2, high(number_of_values)
	ldi temp1, low(number_of_values); hherwertigere Einheit lesen
	rcall eep_read
	sts dwerte, temp1				; im RAM merken
	rcall get_kanaele				; Kanalanzahl lesen
	sts dkanaele, temp1				; im RAM merken
	ldi temp1, 1
	sts dkanal, temp1				; 
	ldi temp1, 0
	sts dawert,temp1				; 
menu0_e:
menu0_done:
	rjmp menu_done
; - - - - Men 1 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
menu1:
	cpi wahl, msek			; Men 1 aktiv?
	brne menu2
	; Menoptionen
menu1a:
	ldi temp1, low(timebase_low)
	ldi temp2, high(timebase_low)
	rcall eep_read			; Wert ist jetzt in temp1
	push temp1
	ldi temp1, 6
	ldi temp2, 1
	rcall lcd_position
	pop temp1
	rcall lcd_number
	rcall lcd_blank
	sbrs flag, taste2		; bei Taste2 aufwrts zhlen bis maximal 240
	rjmp menu1b				; oder mglichereise bei Taste 3 abwrts zhlen bis 1
	; aufwrts zhlen bis maximal maximalwert
	inc temp1
	cpi temp1, 100			; aufwrts zhlen bis maximal 99
	brlo menu1bb
menu1aa:
	ldi temp1, 0			; setze auf 0 falls 99 berschritten
	rjmp menu1bb
menu1b:
	sbrs flag, taste3		; berspringe rjmp, falls Taste 3 gedrckt, dann abwrts zhlen bis 1
	rjmp menu1_done			; abwrtszhlen bis minimal 1
	dec temp1
	cpi temp1, 0
	brpl menu1bb
	ldi temp1, 99
menu1bb:					; Wert merken
	ldi temp2, low(timebase_low)
	ldi temp3, high(timebase_low)
	rcall eep_write			; Wert schreiben
	push temp1
	ldi temp1, 6
	ldi temp2, 1
	rcall lcd_position
	pop temp1
	rcall lcd_number
	rcall lcd_blank
	rcall timebase_set
menu1_done:
	rjmp menu_done			; Option fertig bearbeitet

; - - - - Men 2 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
menu2:
	cpi wahl, mmin			; Men 2 aktiv?
	brne menu3
	; Menoptionen
menu2a:
	ldi temp1, low(timebase_high)
	ldi temp2, high(timebase_high)
	rcall eep_read			; Wert ist jetzt in temp1
	push temp1
	ldi temp1, 6
	ldi temp2, 1
	rcall lcd_position
	pop temp1
	rcall lcd_number
	rcall lcd_blank
	sbrs flag, taste2		; bei Taste2 aufwrts zhlen bis maximal 240
	rjmp menu2b				; oder mglichereise bei Taste 3 abwrts zhlen bis 1
	; aufwrts zhlen bis maximal maximalwert
	inc temp1
	cpi temp1, 100			; aufwrts zhlen bis maximal 99
	brlo menu2bb
menu2aa:
	ldi temp1, 0			; setze auf 0 falls 99 berschritten
	rjmp menu2bb
menu2b:
	sbrs flag, taste3		; berspringe rjmp, falls Taste 3 gedrckt, dann abwrts zhlen bis 1
	rjmp menu2_done			; abwrtszhlen bis minimal 1
	dec temp1
	cpi temp1, 0
	brpl menu2bb
	ldi temp1, 99
menu2bb:					; Wert merken
	ldi temp2, low(timebase_high)
	ldi temp3, high(timebase_high)
	rcall eep_write			; Wert schreiben
	push temp1
	ldi temp1, 6
	ldi temp2, 1
	rcall lcd_position
	pop temp1
	rcall lcd_number
	rcall lcd_blank
	rcall timebase_set
menu2_done:
	rjmp menu_done			; Option fertig bearbeitet

; - - - - Men 3 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
menu3:
	cpi wahl, mint			; Men 3 aktiv?
	brne menu4
	; Menoptionen
menu3a:
	sbrs flag, taste2
	rjmp menu3_done
menu3b:
	rcall timebase_set
	ldi temp1, 10
	ldi temp2, 0
	rcall lcd_position
	lds r3, Zeit_low
	lds r4, zeit_high
	clr r5
	rcall lcd_binout
menu3_done:
	rjmp menu_done			; Option fertig bearbeitet

; - - - - Men 4 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
menu4:
	cpi wahl, mmaxsamp		; Men 4 aktiv?
	brne menu5
	; Menoptionen
menu4a:
	lds temp1, maxzahl
	sbrs flag, taste2		; bei Taste2 aufwrts zhlen bis maximal 240
	rjmp menu4b				; oder mglichereise bei Taste 3 abwrts zhlen bis 1
	; aufwrts zhlen bis maximal maximalwert
	inc temp1
	cpi temp1, maximalwert+1
	brlo menu4bb
	ldi temp1, 1
menu4aa:
	rjmp menu4bb
menu4b:
	sbrs flag, taste3		; berspringe rjmp, falls Taste 3 gedrckt, dann abwrts zhlen bis 1
	rjmp menu4_done
	; abwrtszhlen bis minimal 1
	dec temp1
	brne menu4bb
	ldi temp1, maximalwert
menu4bb:
	sts maxzahl, temp1
	; werte anzeigen
	rcall show_samples
menu4_done:
	rjmp menu_done			; Option fertig bearbeitet
	
; - - - - Men 5 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
menu5:
	cpi wahl, mloeschen		; Men 5 aktiv?
	brne menu6
	; Menoptionen
menu5a:
	sbrs flag, taste2
	rjmp menu5_done
	rcall werte_loeschen
	ldi zl, low(2*mcleared)	; Basisadresse fr Datenrecord: Spalte, Zeile, Text holen, Lowbyte 
	ldi zh,high(2*mcleared)	; Highbyte
	lpm temp1, z+			; Spalte fr LCD-Ausgabe holen
	lpm temp2, z+			; Zeile fr LCD-Ausgabe holen
	rcall lcd_position		; durch Postincrement steht der Zeiger jeweils auf dem nchsten Feld
	rcall lcd_flash_string	; nullterminierten Text ausgeben
menu5_done:
	rjmp menu_done			; Option fertig bearbeitet

; - - - - Men 6 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
menu6:
	cpi wahl, mstopngo		; Men 6 aktiv?
	brne menu7		
	; Menoptionen 6
menu6a:
	sbrs flag, taste2
	rjmp menu6b
	ldi count_low, 1
	ldi count_high,0
	sbr flag, 1<<fblinken	; lst Blinken der LED durch die ISR von Timer0 aus
	ldi zl, low(2*mrunning)	; Basisadresse fr Datenrecord: Spalte, Zeile, Text holen, Lowbyte 
	ldi zh,high(2*mrunning)	; Highbyte
	lpm temp1, z+			; Spalte fr LCD-Ausgabe holen
	lpm temp2, z+			; Zeile fr LCD-Ausgabe holen
	rcall lcd_position		; durch Postincrement steht der Zeiger jeweils auf dem nchsten Feld
	rcall lcd_flash_string	; nullterminierten Text ausgeben
	rjmp menu6_done
menu6b:
	sbrs flag, taste3		; berspringe rjmp, falls Taste 3 gedrckt
	rjmp menu6_done
	cbr flag, 1<<fblinken	; beendet Blinken der LED und den Messzyklus
	ldi zl, low(2*mstoped)	; Basisadresse fr Datenrecord: Spalte, Zeile, Text holen, Lowbyte 
	ldi zh,high(2*mstoped)	; Highbyte
	lpm temp1, z+			; Spalte fr LCD-Ausgabe holen
	lpm temp2, z+			; Zeile fr LCD-Ausgabe holen
	rcall lcd_position		; durch Postincrement steht der Zeiger jeweils auf dem nchsten Feld
	rcall lcd_flash_string	; nullterminierten Text ausgeben
menu6_done:
	rjmp menu_done			; Option fertig bearbeitet

; - - - - Men 7 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
menu7:
	cpi wahl, msend			; Men 7 aktiv?
	brne menu8
	sbrs flag, taste2
	rjmp menu7_done
menu7a:
	; Menoptionen 7
	ldi zl, low(2*msending)	; Basisadresse fr Datenrecord: Spalte, Zeile, Text holen, Lowbyte 
	ldi zh,high(2*msending)	; Highbyte
	lpm temp1, z+			; Spalte fr LCD-Ausgabe holen
	lpm temp2, z+			; Zeile fr LCD-Ausgabe holen
	rcall lcd_position		; durch Postincrement steht der Zeiger jeweils auf dem nchsten Feld
	rcall lcd_flash_string	; nullterminierten Text ausgeben
	rcall werte_transfer
menu7b:
	ldi zl, low(2*msendend)	; Basisadresse fr Datenrecord: Spalte, Zeile, Text holen, Lowbyte 
	ldi zh,high(2*msendend)	; Highbyte
	lpm temp1, z+			; Spalte fr LCD-Ausgabe holen
	lpm temp2, z+			; Zeile fr LCD-Ausgabe holen
	rcall lcd_position		; durch Postincrement steht der Zeiger jeweils auf dem nchsten Feld
	rcall lcd_flash_string	; nullterminierten Text ausgeben
menu7_done:
	rjmp menu_done			; Option fertig bearbeitet

; - - - - Men 8 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
menu8:
	cpi wahl, mLCDclear		; Men 8 aktiv?
	brne menu9
	; Menoptionen
	sbrs flag, taste2
	rjmp menu8_done

	rcall vorgabewerte


	ldi zl, low(2*mset)		; Basisadresse fr Datenrecord: Spalte, Zeile, Text holen, Lowbyte 
	ldi zh,high(2*mset)		; Highbyte
	lpm temp1, z+			; Spalte fr LCD-Ausgabe holen
	lpm temp2, z+			; Zeile fr LCD-Ausgabe holen
	rcall lcd_position		; durch Postincrement steht der Zeiger jeweils auf dem nchsten Feld
	rcall lcd_flash_string	; nullterminierten Text ausgeben
menu8_done:
	rjmp menu_done			; Option fertig bearbeitet

; - - - - Men 9 - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
menu9:
	cpi wahl, mKanal		; Men 9 aktiv?
	brne menu10
	; Menoptionen
menu9a:
	ldi temp1, 6
	ldi temp2, 1
	rcall lcd_position
	rcall get_kanaele
	rcall lcd_number
	rcall lcd_blank
	sbrs flag, taste2		; bei Taste2 aufwrts zhlen bis maximal 4
	rjmp menu9b				; oder mglichereise bei Taste 3 abwrts zhlen bis 1
	; aufwrts zhlen bis maximal maximalwert
	inc temp1
	cpi temp1, maxkanal+1	; aufwrts zhlen bis maximal 4
	brlo menu9bb
menu9aa:
	ldi temp1, 1			; setze auf 1 falls 4 berschritten
	rjmp menu9bb
menu9b:
	sbrs flag, taste3		; berspringe rjmp, falls Taste 3 gedrckt, dann abwrts zhlen bis 1
	rjmp menu9_done			; abwrtszhlen bis minimal 1
	dec temp1
	cpi temp1, 1
	brpl menu9bb
	ldi temp1, maxkanal
menu9bb:					; Wert merken
	ldi temp2, low(kanaele)
	ldi temp3, high(kanaele)
	rcall eep_write			; Wert schreiben
	push temp1
	ldi temp1, 6
	ldi temp2, 1
	rcall lcd_position
	pop temp1
	rcall lcd_number
	rcall lcd_blank
menu9_done:
	rjmp menu_done			; Option fertig bearbeitet
	
; - - - - Men 10  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
menu10:						
	cpi wahl, mWerte		; Men 10 aktiv?
	brne menu11
	; Menoptionen ber die Tasten 2 und 3
	sbrs flag, taste2		; bei Taste2 aufwrts zhlen bis maximal 240
	rjmp menu10b				; oder mglichereise bei Taste 3 abwrts zhlen bis 1
menu10a: ;(taste 2)
	; Zhlerstand prfen, wenn Kanalanzahl addiert wird
	lds temp1, dawert			; Aktuelle Wertenummer bei der LCD-Darstellung
	lds temp2, dkanaele			; Anzahl Eingangskanle fr aktuelle Reihe
	add temp1, temp2			; beide addieren und  vergleichen mit
	lds temp2, dwerte			; bisherige Anzahl Messwerte gesamt
	inc temp2					; plus eins
	cp  temp1, temp2			; ist dawert + dkanaele <= dwerte?
	
	brlo menu10d				; noch unter der obergrenze, dann zur Ausgabesequenz
	; sonst berspringen
	rjmp menu10_done			; Obergrenze erreicht, also nix machen
menu10b: ;(taste 3)
	sbrs flag, taste3			; berspringe rjmp, falls Taste 3 gedrckt, dann abwrts zhlen bis 1
	rjmp menu10_done			; abwrtszhlen bis minimal 1
	; zhlerstand prfen, ob noch kanalzahl abgezogen werden kann
	lds temp1, dawert			; Aktuelle Wertenummer bei der LCD-Darstellung
	lds temp2, dkanaele			; Anzahl Eingangskanle fr aktuelle Reihe
	sub temp1, temp2			; dawert - dakanaele >= 0
	sub temp1, temp2			; dawert - dakanaele >= 0
	brcc menu10c				; geht noch, dann abziehen und zur Ausgabesequenz
	ldi temp1, 0
menu10c:
	sts dawert, temp1			; geht nicht mehr, dann = 0 setzen und zur Ausgabesequenz
menu10d: ;Ausgabesequenz
	ldi temp1, 0x07				; Cursorposition fr Ausgabe der Nummer des Messwertsets
	ldi temp2, 0
	rcall lcd_position
	lds temp1, dawert			; Nummer des aktuellen Messwertsets
	rcall lcd_number
	rcall lcd_blank

	ldi temp1, 0x0b				; ersten Kanal ausgeben
	ldi temp2, 0
	rcall lcd_wert				; nchsten Wert ausgeben und Zhler erhhen	
	
	lds temp1, dkanaele
	cpi temp1, 2				; falls dkanaele < 2 dann 
	brlo menu10_done			; fertig

	ldi temp1, 0x00				; zweiten Kanal ausgeben
	ldi temp2, 1
	rcall lcd_wert				; nchsten Wert ausgeben und Zhler erhhen	
	
	lds temp1, dkanaele
	cpi temp1, 3				; falls dkanaele < 3 dann 
	brlo menu10_done			; fertig

	ldi temp1, 0x05				; dritten Kanal ausgeben
	ldi temp2, 1
	rcall lcd_wert				; nchsten Wert ausgeben und Zhler erhhen	
	
	lds temp1, dkanaele
	cpi temp1, 4				; falls dkanaele < 4 dann 
	brlo menu10_done			; fertig

	ldi temp1, 0x0A				; vierten Kanal ausgeben
	ldi temp2, 1
	rcall lcd_wert				; nchsten Wert ausgeben und Zhler erhhen	

menu10_done:
	rjmp menu_done			; Option fertig bearbeitet
; - - - - Men 11  - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
menu11:						; nchste Programmausbaustufe
;	rjmp menu_done			; Option fertig bearbeitet
menu_done:					; alle aktiven Tasten-Flags zurcksetzen
	andi flag, ~((1<<keypressed) | (1<<taste1) | (1<<taste2) | (1<<taste3))

	pop zh
	pop zl
	ret
	
rs_befehl:					; RS232-Befehl decodieren und ausfhren
	rcall serin_echo		; erstes Zeichen des neuen Befehls einlesen
	cpi temp1, 'e'			; und auf die gltigen Flle berprfen.
	brne job01
job00:
	; setze das Zeitintervall aus dem EEPROM
	rcall timebase_set
	rjmp job99
	; LCDCursor home
job01:
	cpi temp1, 'h'
	brne job02
	rcall lcd_home
	rjmp job99
	; LCD clear
job02:
	cpi temp1, 'c'
	brne job03
	rcall lcd_clear
	rjmp job99
	;LCD initialize
job03:
	cpi temp1, 'i'
	brne job04
	rcall lcd_init
	rjmp job99
	; setze das Zeitintervall in low
job04:
	cpi temp1, 's'
	brne job05
	rcall v24_in_number			; zweistellige Zahl als Bytewert holen
	ldi temp2, low(timebase_low)
	ldi temp3, high(timebase_low)
	rcall eep_write
;	rcall lcd_number
	ldi temp1,10
	rcall delay_nx5ms
	rjmp job99
	; setze das Zeitintervall in high
job05:
	cpi temp1, 'm'
	brne job06
	rcall v24_in_number			; zweistellige Zahl als Bytewert holen
	ldi temp2, low(timebase_high)
	ldi temp3, high(timebase_high)
	rcall eep_write
;	rcall lcd_number
	ldi temp1,10
	rcall delay_nx5ms
	rjmp job99
	; Messsequenz starten
job06:
	cpi temp1, 'g'
	brne job07
	ldi count_low, 1
	ldi count_high,0
	sbr flag, 1<<fblinken		; lst Blinken der LED durch die ISR von Timer0 aus
	rjmp job99
	; Manueller Abbruch der Messsequenz
job07:
	cpi temp1, 'b'
	brne job08
	cbr flag, 1<<fblinken		; stoppt Blinken der LED
	rjmp job99
	; Anzeige der Anzahl Messwerte im EEPROM und im Register anzahlwerte
job08:
	cpi temp1, 'a'
	brne job09
	ldi temp1,10
	rcall delay_nx5ms
	ldi temp2, high(number_of_values)
	ldi temp1, low(number_of_values)	; hherwertigere Einheit lesen
	rcall eep_read
	rcall v24_number
;	ldi temp1, ' '
;	rcall serout
	rcall v24_lfcr
	mov temp1, anzahlwerte
	rcall v24_number
	rcall v24_lfcr
	rjmp job99
	; sende die Anzahl Werte, die Zeitbasis und die Werte selbst ber die RS232
job09:
	cpi temp1, 't'
	brne job10
	rcall werte_transfer
	rjmp job99
	; Setze die Maximalzahl automatisch aufgenommener Messwerte
job10:
	cpi temp1, 'o'
	brne job11
	rcall maxzahl_setzen
	rjmp job99
	; Lsche die Messwerte im EEPROM (eigentlich nur die Anzahl Werte)
job11:
	cpi temp1, 'l'
	brne job12
	rcall werte_loeschen
	rjmp job99
	; Setze die Maximalzahl automatisch aufgenommener Messwerte
job12:
	cpi temp1, 'O'
	brne job13
	rcall maxzahl_abfragen
	rjmp job99
	; Setze die Anzahl von Messeingngen (Anzahl Kanle)
job13:
	cpi temp1, 'k'
	brne job14
	rcall kanalzahl_setzen
	rjmp job99

job14:
	cpi temp1, 'K'
	brne job15
	rcall kanalzahl_abfragen
	rjmp job99

job15:
job99:
	ldi temp1, 0x0a
	rcall serout
	ldi temp1, 0x0d		; neue Zeile
	rcall serout			
	ret

; ********************************************************
; ****************** IRQ - Jobroutinen *******************
; ********************************************************
; Zeitsteuerung
;
Timer1_oc1a:
	push temp1						; temp1 sichern
	in temp1, sreg					; Statusregister sichern
	push temp1
	push temp2
									; Timer1 erzeugt alle 8 mal 800 = 6400 Systemtakte einen IRQ
	lds temp1, grubs				; Ablaufsteuerung  wie bei Jiffies nur feinere Zeitscheiben
	inc temp1						; 25 Grubs ergeben eine 50stel-Sekunde
	cpi temp1, 25					; Diese Sequenz wird alle 0,80 ms durchlaufen
	breq weiter_mit_jiffies			; 1250 mal Durchlaufen ergibt eine Sekunde
	sts grubs, temp1				; Bei Werten unter 25 werden hier kurze periodische 
									; Vorgnge wie Tastenabfragen eingebaut, von denen jeder
									; eine Zeitscheibe von eben knapp 0,80 ms erhlt
									; bei 8MHz sind das 6400 Takte von denen aber die Takte der 
									; ISR abzuziehen sind.
; *************************************************************************************************
; Tastenabfrage, Reaktion auf Tasten, wird im Abstand von 50stel Sekunden aufgerufen
; *************************************************************************************************
	; Job 1   Zeitscheibe 1: Tastenabfrage +++ wird im Abstand von 50stel Sekunden aufgerufen   +++
grb01:
	lds temp1, grubs
	cpi temp1, 1					; 
	brne grb02
	; Taste 1 gedrckt, wenn taste1-bit=0  ........................................................
	sbic tastenpin, taste1			; berspringe nchsten Befehl, wenn Bit = 0
	rjmp t1r						; bit = 1 -> Taste nicht gedrckt
t1p:
	inc t1							; Zhler fr Taste 1 erhhen
	cpi t1, longtime+1				; Langzeit Tastendruck	
	brlo t1_ok
	ldi t1,longtime
t1_ok: 
	rjmp t1_fertig
t1r:
	clr t1							; Zhler fr Taste 1 auf 0, wenn nicht gedrckt
t1_fertig:

	; Taste 2 gedrckt, wenn taste2-bit=0 .........................................................
	sbic tastenpin, taste2			; berspringe nchsten Befehl, wenn Bit = 0
	rjmp t2r						; bit = 1 -> Taste nicht gedrckt
t2p:
	inc t2							; Zhler fr Taste 2 erhhen
	cpi t2, longtime+1				; Langzeit Tastendruck	
	brlo t2_ok
	ldi t2,0
t2_ok: 
	rjmp t2_fertig
t2r:
	clr t2							; Zhler fr Taste 1 auf 0, wenn nicht gedrckt
t2_fertig:

	; Taste 3 gedrckt, wenn taste3-bit=0 ........................................................
	sbic tastenpin, taste3			; berspringe nchsten Befehl, wenn Bit = 0
	rjmp t3r						; bit = 1 -> Taste nicht gedrckt
t3p:
	inc t3							; Zhler fr Taste 3 erhhen
	cpi t3, longtime+1				; Langzeit Tastendruck	
	brlo t3_ok
	ldi t3,0
t3_ok: 
	rjmp t3_fertig
t3r:
	clr t3							; Zhler fr Taste 1 auf 0, wenn nicht gedrckt
t3_fertig:

grb01_done:
	rjmp end_isr

	; Job 2   Zeitscheibe 2: Tastenaktionen +++wird im Abstand von 50stel Sekunden aufgerufen+++
grb02:
	
t1k:
	lds temp1, grubs
	cpi temp1, 2					; 
	brne grb03
	; Taste 1 war kurz gedrckt, wenn t1=3
	cpi t1,3
	brne t2k
	cbr flag, (1<<mns)				; Mentaste mindestens einmal gedrckt, mns-Flag rcksetzen
	inc wahl
	cpi wahl, mlast
	brlo t1k_0
	ldi wahl, 1
t1k_0:
	sbr flag, (1<<keypressed) | (1<<taste1)	; Tastenflags setzen
	rjmp grb02_done

t2k:
	cpi t2, 3 						; entsprechenden Kanal aufwrtszhlen
	brne t3k
	sbrc flag, mns					; falls die Menauswahl schon luft, berspringe rjmp-befehl
	rjmp grb02_done
	sbr flag, (1<<keypressed) | (1<<taste2)	; Tastenflags setzen
	rjmp grb02_done

t3k:
	cpi t3, 3 						; entsprechenden Kanal abwrtszhlen
	brne grb03
	sbrc flag, mns					; falls die Menauswahl schon luft, berspringe rjmp-befehl
	rjmp grb02_done
	sbr flag, (1<<keypressed) | (1<<taste3)	; Tastenflags setzen

grb02_done:
	rjmp end_isr
grb03:
	rjmp end_isr
	; --------------------- IRQ - Zeitscheiben-Aufgaben  fertig ---------------------------------

weiter_mit_jiffies:					; wird im Abstand von 50stel Sekunden aufgerufen
	clr temp1
	sts grubs, temp1
	lds temp1, jiffies				; Jiffies laden
	inc temp1						; Jiffies erhhen
	cpi temp1, 50					; mit 50 vergleichen
	breq weiter_mit_sekunden		; 50 erreicht, dann mit Sekunden weiter
	sts jiffies, temp1				; noch keine 50, -> speichern und


; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
; 	LED blinken lassen, wenn Flag fblinken gesetzt ist
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
	sbrs flag, fblinken				; Messung luft, wenn fblinken = 1
	rjmp loeschen					; Messung luft nicht, LED aus!
led_blinken:
	lds temp1, jiffies
	cpi temp1, 49
	brlo loeschen					; LED1 aufblitzen lassen (1/50 Sekunde)
	sbi led_port, led1				; LED1 einschalten beim 49. Durchlauf
	rjmp end_isr					; bit_fertig
loeschen:
	cbi led_port, led1				; so bald jiffies = 0, wird die LED wieder ausgeschaltet
	rjmp end_isr					; fertig
									
weiter_mit_sekunden:				; es ist der wert 50 erreicht
	clr temp1						; Jiffies lschen 
	sts jiffies, temp1				; abspeichern


; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
;	Messen ist aktiv, wenn LED blinkt
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

bit_fertig:
	sbrs flag, fblinken				; Messung luft, wenn fblinken = 1
	rjmp getvalue_done				; Messung luft nicht, also keine Auswertung!
	; count_high:low -1
	sbiw R26, 1 					; Countdownzhler fr Zeitintervall verringern
	; Count_high:low = 0?
	breq getvalue					; auf Null gezhlt? dann wird's Zeit fr eine Messung
	rjmp getvalue_done				; nicht Null, dann im Moment keine Messung ntig
getvalue:
	sbr flag, 1<<fgetvalue			; Erst mal Flag setzen, dass neue Messung erfolgen muss
									; ja, dann Wert holen  (das macht die Jobschleife!)
									; wertezhler kleiner als Maximum?
	lds temp2, maxzahl
	cp anzahlwerte, temp2
	brlo set_countdown				;
									; nein, fblinken aus, messen beendet
	cbr flag, 1<<fblinken
	rjmp getvalue_done
set_countdown:
									; ja, countdown neu setzen und weiter
	lds count_low, zeit_low			; Der Wert steht als Low/Highbyte im SRAM
	lds count_high, zeit_high
	inc anzahlwerte					; wir haben einen Messwert mehr

getvalue_done:						; Messwertbehandlung beendet

; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
;	Ganz normal weiter in der Zeitrechnung
; ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

	lds temp1, sekunden
	inc temp1						; sekunden erhhen

	cpi temp1, 60					; 60 Sekunden voll?
	breq weiter_mit_minuten			; dann Minutenbertrag
	sts sekunden, temp1				; nein, noch nicht
	rjmp fertig						; dann fertig

weiter_mit_minuten:
	clr temp1						; Sekunden lschen und
	sts sekunden, temp1				; die Null merken

	lds temp1, minuten				; wie bei Sekunden
	inc temp1	
	cpi temp1, 60		
	breq weiter_mit_stunden
	sts minuten, temp1
	rjmp fertig

weiter_mit_stunden:
	clr temp1
	sts minuten, temp1

	lds temp1, stunden				; wie bei Sekunden
	inc temp1
	cpi temp1, 24
	breq tag_fertig
	sts stunden, temp1
	rjmp fertig
	
tag_fertig:
	clr temp1
	sts stunden, temp1				; Zeitrechnung startet von vorne

fertig:

end_isr:
	pop temp2
	pop temp1
	out sreg, temp1					; status register restaurieren
	pop temp1						; temp1 zurck
	reti							; irq-routine beenden


.nolist
.include "arithmetik.asm"
.include "adc_m8_inc.asm"
.include "lcd-routines-bib.asm"
.include "rs232inc.asm"
.include "eeprom_512_inc.asm"
.list


; ********************************************************
; ********************************************************
; Job-Proceduren
; ********************************************************
; ********************************************************
; Zeitintervall setzen
timebase_set:
	ldi temp1, 10
	rcall delay_nx5ms
	ldi temp2, high(timebase_high)
	ldi temp1, low(timebase_high)	; hherwertigere Einheit lesen
	rcall eep_read
;	rcall lcd_number
	clr R23
	mov R22, temp1
	clr R25
	ldi R24, 60
	rcall mulx16					; hherwertigere Einheit mal 60
	ldi temp2, high(timebase_low)
	ldi temp1, low(timebase_low)	; niederwertigere Einheit lesen
	rcall eep_read
;	rcall lcd_number
	clr temp2
	add R2, temp1
	adc R3, temp2					; zum Produktwert addieren
	sts zeit_low, R2
	sts zeit_high, R3				; Zeitintervall abspeichern und
	mov count_low, r2
	mov count_high, r3				; in countdown-zhler schreiben
	sbrc flag, keypressed			; falls Taste am Datenlogger nicht gedrckt war
	rjmp tbs_05						; rjmp berspringen. Falls Taste gedrckt, keine 
									; Ausgabe ber RS232
	mov r4,r3						; Werte in AkkuB justieren
	mov r3,r2
	rcall v24_binout				; auf rs232 ausgeben
tbs_05:
	ret


; Aktuellen Messwert ins EEPROM, Zeiger und Anzahlwerte nachfhren, Anzahl Kanle berchsichtigen
werte_sichern: 						; y-Pointer zeigt auf nchste freie EEPROM-Adresse
	push R15

	rcall get_kanaele				; Anzahl steht in temp1
	mov rkanaele, temp1
	clr R15
wesi_00:
	mov temp1, R15
	rcall adc_set_channel			; Kanal einstellen	
	ldi temp1, 3					; hole den Mittelwert von 2 hoch 3 = 8
	rcall adc_get					; Einzelmessungen vom ADC
	mov temp1, r3					;
	mov temp2, yl
	mov temp3, yh
	rcall eep_write					; Lowbyte im EEPROM ablegen 
	adiw R28, 1

	mov temp1, r4
	mov temp2, yl
	mov temp3, yh
	rcall eep_write					; Highbyte im EEPROM ablegen 
	adiw R28, 1
	
	mov temp1, anzahlwerte			; wurde in der ISR zu Timer1 bereits erhht
	sts dwerte, temp1				; Werteanzahl fr LCD-Ausgabe merken
	ldi temp2, low(number_of_values)
	ldi temp3, high(number_of_values)
	rcall eep_write					; Anzahl der Werte merken im EEPOROM

	inc r15
	dec rkanaele
	brne wesi_00

	cbr flag, 1<<fgetvalue			; Flag fr neuen Messwert rcksetzen

	; werte ins LCD
	ldi temp1, 0
	ldi temp2, 1					; in Spalte0, Zeile2 den Messwert
	rcall lcd_position
	mov temp1, anzahlwerte
	rcall lcd_number				; Messertnummer ausgeben und
	ldi temp1, ' '
	rcall lcd_data
	rcall lcd_binout
	ldi temp1, ' '
	rcall lcd_data

	pop R15
	ret

; Anzahlwerte, Zeitdelay und Werte an die RS232
werte_transfer: 
	push yl
	push yh
	push anzahlwerte
	in temp1, sreg
	push temp1

	ldi temp1, 10
	rcall delay_nx5ms
	ldi temp2, high(number_of_values)
	ldi temp1, low(number_of_values)	; hherwertigere Einheit lesen
	rcall eep_read
	mov anzahlwerte, temp1			; als Zhler merken
	rcall v24_number				; auf rs232 ausgeben
	rcall v24_lfcr					; CR ausgeben
	rcall delay5ms
	tst anzahlwerte					; Anzahlwerte = 0?
	breq wt02						; ja, -> fertig

	ldi temp2, high(kanaele)
	ldi temp1, low(kanaele)			; Anzahl Mess-Kanaele lesen
	rcall eep_read					; ausgeben
	rcall v24_number				; auf rs232 
	rcall v24_lfcr					; CR ausgeben
	rcall delay5ms					; Warten auf langsamen PC

	ldi temp2, high(timebase_high)
	ldi temp1, low(timebase_high)	; hherwertigere Zeitangabe lesen
	rcall eep_read
	clr R23			
	mov R22, temp1					; nach Faktor 1 kopieren
	clr R25
	ldi R24, 60						; Faktor2 = 60
	rcall mulx16					; Hoher Zeitwert mal 60
	ldi temp2, high(timebase_low)
	ldi temp1, low(timebase_low)	; niederwertigere Zeitangabe lesen
	rcall eep_read
	clr temp2						; 0 nach temp2
	add R2, temp1					; niedriger Zeitwert + LSB vom Produkt
	adc R3, temp2					; MSB vom Produkt und evtl. bertrag (Carry) bercksichtigen
	mov r4,r3						; Anzahl von Sekunden (Minuten)
	mov r3,r2						; LSB und MSB in AkkuB richtig justieren
	rcall v24_binout				; auf rs232 ausgeben
	rcall v24_lfcr					; CR ausgeben
	rcall delay5ms

	ldi yl, low(werteliste)			; y-Pointer setzen auf
	ldi yh, high(werteliste)		; AdressenBasis fr Werteliste im EEPROM	
	clr R5							; HSB in AkkuB vorsorglich lschen (nur 2 Byte ntig)
wt01:
	mov temp1, yl
	mov temp2, yh					; Adresse im EEPROM
	rcall eep_read					; Wert Lowbyte lesen
	mov R3, temp1					; nach AkkuB Low kopieren fr Ausgaberoutine
	adiw y, 1						; Zeiger ins EEPROM erhhen
	mov temp1, yl
	mov temp2, yh					; Adresse im EEPROM
	rcall eep_read					; Highbyte lesen
	mov R4, temp1					; nach AkkuB middle kopieren
	adiw y, 1						; Zeiger ins EEPROM fr nchsten Durchgang erhhen
	rcall v24_binout				; AkkuB auf rs232 ausgeben
	rcall v24_lfcr					; Trennzeichen LF+CR
	rcall delay5ms
	dec anzahlwerte					; Anzahl herunterzhlen
	brne wt01						; wiederhole solange anzahlwerte noch grer 0
wt02:
	pop temp1
	out sreg,temp1
	pop anzahlwerte
	pop yh
	pop yl
	ret


kanalzahl_setzen:
	push temp2
	push temp3
	in temp3,sreg
	push temp3
	
	rcall v24_read_int				; Zahl einlesen, Wert steht in AkkuB (R5:R3)
	and R4, r4						; Ist R4 = 0?
	brne mk01						; nein, dann Fehler, weil Eingabe >= 256
	mov temp1, R3
	cpi temp1, maxkanal+1			; Ist R3 (LSB) > maxkanal (4)?
	brsh mk01						; ja, dann Fehler, weil Eingabe zu gro
	tst r3							; ist R3 = 0?
	brne mk00a						; nein, dann OK
	ldi temp1, 1					; Null geht nicht
	mov r3, temp1					; wir setzen R3 = 1
mk00a:
	ldi temp2, low(kanaele)			; Merke die Obergrenze fr Messwerte im eeprom
	ldi temp3, high(kanaele)
	rcall eep_write			; Wert schreiben
mk00:
	pop temp3
	out sreg, temp3
	pop temp3
	pop temp2
	ret
mk01:
	ldi zl, low(2*str_fehler)		; y-Zeiger auf Adresse der Fehlermeldung setzen
	ldi zh, high(2*str_fehler)
	rjmp mk00


maxzahl_setzen:
;	ldi temp1, 10
;	rcall delay_nx5ms
;	ldi zl, low(2*str_test)
;	ldi zh, high(2*str_test)		; Zeiger auf Eingabe-Prompt
;	rcall lcd_flash_string			; String ausgeben
;	ldi temp1, 10
;	rcall delay_nx5ms
	rcall v24_read_int				; Zahl einlesen, Wert steht in AkkuB (R5:R3)
;	rcall lcd_binout					; Zeilenvorschub und Wagenrcklauf
	and R4, r4						; Ist R4 = 0?
	brne ms01						; nein, dann Fehler, weil Eingabe >= 256
	mov temp1, R3
	cpi temp1, maximalwert+1		; Ist R3 (LSB) > 240?
	brsh ms01						; ja, dann Fehler, weil Eingabe zu gro
	tst r3							; ist R3 = 0?
	brne ms00a						; nein, dann OK
	ldi temp1, 1					; Null geht nicht
	mov r3, temp1					; wir setzen R3 = 1
ms00a:
	sts maxzahl, R3					; Merke die Obergrenze fr Messwerte im SRAM
;	ldi zl, low(2*str_registriert)	; mit 
;	ldi zh, high(2*str_registriert)	; Besttigungstext
ms00:
;	rcall v24_flash_string
	ret
ms01:
	ldi zl, low(2*str_fehler)		; y-Zeiger auf Adresse der Fehlermeldung setzen
	ldi zh, high(2*str_fehler)
	rjmp ms00


maxzahl_abfragen:
	ldi temp1, 10
	rcall delay_nx5ms
	lds temp1, maxzahl
	rcall v24_number				; registrierten Wert zurckmelden
	rcall v24_lfcr					; neue Zeile
	ret


kanalzahl_abfragen:
	ldi temp1, 10
	rcall delay_nx5ms
	rcall get_kanaele
	rcall v24_number				; registrierten Wert zurckmelden
	rcall v24_lfcr					; neue Zeile
	ret




werte_loeschen:						; Es gengt, die Anzahl Werte auf 0 zu setzen
;	ldi yl, low(werteliste)			; den EEPROM-Speicher knnte man auch lschen
;	ldi yh, high(werteliste)		; indem wir in jede Zelle eine 0xFF schreiben
	rjmp wl01						; aber wir lassen das, um Zeit zu sparen
wl00:
	mov temp2, yl					; Umschaufeln der EEPROM-Adresse in die bergaberegister
	mov temp3, yh					; fr Low- und Highbyte
	ldi temp1, 0xff					; Wert fr gelschte EEPROM-Zelle ist 0xFF = 0b11111111
	rcall eep_write					; ins EEPROM schreiben
	adiw r28, 1						; es folgt der Vergleich ber 2 Bytes
	ldi temp1, high(werteliste + 2 * maximalwert) ; der hherwertige Teil der Obergrenze muss in ein
	cpi yl, low(werteliste + 2 * maximalwert); Register geladen werden, weil ein Konstantenvergleich
	cpc yh, temp1					; mit Bercksichtigung des Carry-Bits nicht existiert
	brlo wl00						; Die Branch-Befehle sind normal einsetzbar
wl01:								; 
	ldi temp1, 0
	mov anzahlwerte, temp1			; 0 Werte im Registerbereich und 
	ldi temp2, low(number_of_values)
	ldi temp3, high(number_of_values)
	rcall eep_write					; auch im EEPROM
	ldi yl, low(werteliste)			; Wichtig ist das Rcksetzen des Zeigers auf
	ldi yh, high(werteliste)		; die Werteliste
wl02:
	ret


show_menu:
	push temp1
	in temp1, sreg
	push temp1
	push temp2
	push temp3
	
	rcall lcd_clear					; LCD Lschen
	mov temp1, wahl					; Mennummer nach temp1
	ldi temp3, 5
	clr temp2						; MSB lschen
show_m0:
	lsl temp1						; links schieben ins Carry (mal 2)
	rol temp2						; Carry ins MSB schieben
	dec temp3						; das Ganze 5 mal ergibt
	brne show_m0					; wahl mal 32, das ist der offset zum Text
	ldi zl, low(2*welcome)			; Basisadresse der Mentexte nach Register Z
	ldi zh, high(2*welcome)
	add zl, temp1					; LSBs addieren
	adc zh, temp2					; MSBs mit Carry addieren
	rcall lcd_flash_string			; erste Zeile ausgeben
	ldi temp1, 0					; Spalte 1
	ldi temp2, 1					; Zeile 2
	rcall lcd_position				; Cursor an Beginn der 2. Zeile
	rcall lcd_flash_string			; zweite Zeile ausgeben

	pop temp3
	pop temp2
	pop temp1
	out sreg, temp1
	pop temp1
	ret


show_samples:
	push temp1
	push temp2
	in temp2,sreg
	push temp2

	ldi temp1, 13
	ldi temp2, 0
	rcall lcd_position
	mov temp1,anzahlwerte
	rcall lcd_number
	rcall lcd_blank
	ldi temp1, 6
	ldi temp2, 1
	rcall lcd_position
	lds temp1,maxzahl
	rcall lcd_number
	rcall lcd_blank

	pop temp2
	out sreg, temp2
	pop temp2
	pop temp1
	ret


vorgabewerte:
	push temp1
	push temp2
	push temp3
	in temp3,sreg
	push temp3

	ldi temp1, sec_preset			; Sekundenvoreinstellung
	ldi temp2, low(timebase_low)
	ldi temp3, high(timebase_low)
	rcall eep_write
	ldi temp1, min_preset			; Minutenvoreinstellung
	ldi temp2, low(timebase_high)
	ldi temp3, high(timebase_high)
	rcall eep_write
	rcall timebase_set				; Intervalllnge der Pause berechnen
	ldi temp1, 1					; Anzahl Messeingnge Voreinstellung
	ldi temp2, low(kanaele)
	ldi temp3, high(kanaele)
	rcall eep_write

	pop temp3
	out sreg, temp3
	pop temp3
	pop temp2
	pop temp1
	ret


get_kanaele:
	push temp2
	push temp3
	in temp3,sreg

	ldi temp1, low(kanaele)
	ldi temp2, high(kanaele)
	rcall eep_read			; Wert ist jetzt in temp1

	out sreg, temp3
	pop temp3
	pop temp2
	ret


LCD_wert:
	push zl
	in zl, sreg
	push zl
	push zh
	push r3
	push r4
	push r5

	rcall lcd_position			; und dann den Cursor auf Postion
	ldi zl,  low(werteliste)
	ldi zh, high(werteliste)
	lds temp1, dawert			; Jeder Wert ist an 2 EEPROM-Adressen abgelegt
	ldi temp2, 0				; wir verdoppeln daher die relative Addresse in dawert
	mov r5, temp2				; MSB von AccuB schon mal 0 setzen
	lsl temp1
	rol temp2
	add zl, temp1				; dann addieren wir die relative Adresse zum Start
	adc zh, temp2				; der Werteliste
	mov temp1, zl				; EEPROM-adressieren in temp1/temp2
	mov temp2, zh
	rcall eep_read				; und lesen das Lowbyte des ersten gespeicherten Messwerts
	mov R3, temp1				; das wir gleich ins LSB von AccuB schieben
	adiw z, 1					; 1 addieren und wieder 
	mov temp1, zl				; EEPROM-adressieren in temp1/temp2
	mov temp2, zh
	rcall eep_read				; und lesen das Highbyte des ersten gespeicherten Messwerts
	mov R4, temp1
	lds temp1, dawert			; Speicherzeiger erhhen
	inc temp1
	sts dawert, temp1			; und merken
	rcall lcd_binout			; Wert ausgeben
	rcall lcd_blank
	pop r5
	pop r4
	pop r3
	pop zh
	pop zl
	out sreg, zl
	pop zl

	ret 						; und zurck


number_blank:
	rcall v24_number				; Messertnummer ausgeben und
	ldi temp1, ' '
	rjmp serout
	

delay_nx5ms:
	push temp2
	in temp2, sreg
nx5_00:
	rcall delay5ms
	dec temp1
	brne nx5_00
	out sreg, temp2
	pop temp2
	ret

	
str_registriert:
;        0123456789ABCDEF0123456789A
	.db " wurde eingestellt. ", 0

str_fehler:
	.db 0x0a, 0x0d, " F E H L E R !!! Wert groesser 240 !!", 0

str_prompt01:
	.db 0x08, "Bitte eine Zahl groesser 0 aber kleiner 240 eingeben: ",0

str_test:
	.db "Test ", 0

; ********************************************************
; Mentexte
; ********************************************************
welcome: 
m_string:
; 0      0123456789ABCDE  F 
	.db "Datenlogger M8 ",0
	.db "MENUEWAHL -> T1",0	
; 1      0123456789ABCDE  F 
	.db "PAUSE SEKUNDEN:",0
	.db "  vT3 ___ T2^  ",0
; 2      0123456789ABCDE  F 
	.db "PAUSE MINUTEN: ",0
	.db "  vT3 ___ T2^  ",0
; 3      0123456789ABCDE  F 
	.db "INTERVALL ___ s",0
	.db "   T2 = GO     ",0
; 4      0123456789ABCDE  F 
	.db "WERTE MAX AKT__",0
	.db "  vT3 ___ T2^  ",0
; 5      0123456789ABCDE  F 
	.db "WERTE LOESCHEN ",0
	.db "   T2 = GO     ",0
; 6      0123456789ABCDE  F 
	.db "MESSREIHE      ",0
	.db "T2=GO  T3=STOPP",0
; 7      0123456789ABCDE  F 
	.db "WERTE SENDEN   ",0
	.db "T2 = GO        ",0
; 8      0123456789ABCDE  F 
	.db "DEFAULTZEIT/KAN",0
	.db "   T2 = GO     ",0
; 9      0123456789ABCDE  F 
	.db "MESSKANAELE:   ",0
	.db "  vT3 ___ T2^  ",0
;10      0123456789ABCDE  F 
	.db "WERTE: ___ ____",0
	.db "____ ____ ____ ",0

; ********************************************************
; Meldungen
; ********************************************************
mrunning:
		.db 10,0,"-AN- ",0

mstoped:
		.db 10,0,"-AUS-",0

msending:
		.db 8,1,"SENDING",0

msendend:
		.db 8,1,"       ",0

mcleared:
		.db 3,1," GELOESCHT ",0

mset:	.db 3,1,"  GESETZT  ",0

.exit




